function [ndata] = savpivots(data,alldata,ori,termi,idx)
%This function is an automated way to extract the pivot list, rotated
%pivot list and pivot properties of a set of classified synapses.  Used to
%be called synpivots
%Synatax:   [ndata] = savpivots(data,ori,termi,idx);
%Input:     data = punc_data data structure used for the classification
%           alldata = all_data data structure from punc_locodist
%           ori = the origin or central pivot used for the analysis
%           termi = the termini for the analysis, really only used for the
%                   generation of the rotated pivots
%           idx = the index of the synaptic subset
%Output:    ndata = a structure with the fields, pivots, properties, and
%                   summary.  The layers of the structure match the layer
%                   structure of the data.  Note: filename is carried over
%                   from data.

filenames = data(1).filenames;  %pull out the channel names
pathnames = uigetdir2('','Directory to save the files');       %get the directory where you want to save into.
tverts = data(ori).termi;   %grab the terminal vertices
tprops = data(ori).termi_prop;   %grab the terminal properties

h = waitbar(0,'Processing Properties of: ');    %initialize progress bar.
for i = 1:size(filenames,2)     %go through the channels
    %initiate
    sdata = [];
    vert_tmp = [];
    prop_tmp = [];
    %process filename
    strmask = isspace(filenames{i});
    spcloc = find(strmask==1);
    filename = strtrim(filenames{i}(min(spcloc):max(spcloc)));
    waitbar(i/size(filenames,2),h,['Processing Properties of: ' filename]);   %update progress
    if i==ori   %process for data at the origin
        %get self data (self=ori)
        vert_tmp = data(ori).vertices;    %all ori vertices
        vert_tmp(idx,:) = [];             %all relavent vertices
        c1 = vert_tmp;                    %store this for later use
        prop_tmp = flatview2(data(ori).vert_prop,'max');   %all of the ori properties
        prop_tmp(idx,:) = [];         %all of the relavent vertices
    else        %not the origin
        vert_tmp = tverts(:,:,i);    %all ori vertices
        vert_tmp(idx,:) = [];             %all relavent vertices
        if i==termi
            c2 = vert_tmp;           %store this for later use
        end
        vert_tmp(isnan(vert_tmp(:,1)),:) = [];  %remove nan lines
        prop_tmp = tprops(:,:,i);   %all of the ori properties
        prop_tmp(idx,:) = [];         %all of the relavent vertices
        prop_tmp(isnan(prop_tmp(:,1)),:) = [];   %remove nan lines
    end
    %calculate the summary data
    %no outliers
    out = std(prop_tmp(:,2))*5;     %5 times the standard deviation should do it.
    [x,y] = find(prop_tmp(:,2)>out);        %find the outliers.
    prop_tmp(x,:) = [];             %remove the outliers
    sdata(1,:) = mean(prop_tmp);
    sdata(2,:) = std(prop_tmp);
    sdata(3,:) = size(prop_tmp,1);
    sdata(4,:) = median(prop_tmp);
    %save the ori data
    sav2csv(vert_tmp,['vert_',filename,'_',num2str(size(vert_tmp,1))],pathnames);    %save the vertices
    sav2csv(prop_tmp,['prop_',filename],pathnames);     %save the properties
    sav2csv(sdata,['sum_',filename],pathnames);     %save the properties
    %store to ndata
    ndata(i).pivots = vert_tmp;
    ndata(i).props = prop_tmp;
    ndata(i).sum = sdata;
end
close(h);   %close progress bar

%now get the rotated vertices
t_out = [];     %initiate
h1 = waitbar(0,'Rotating Vertices of: ');    %initialize progress bar
parpool      %lets go massive
for j = 1:size(filenames,2)
    if j~=ori && j~=termi
        %process filename
        strmask = isspace(filenames{j});
        spcloc = find(strmask==1);
        filename = strtrim(filenames{j}(min(spcloc):max(spcloc)));
        waitbar(j/size(filenames,2),h1,['Rotating Vertices of: ' filename]);   %update progress
        c3 = alldata(ori).all_termi(:,:,j);     %get the third point
        [chan,t_tmp] = syn_align_loco(c1,c2,c3);
        [chan1] = flat_verts(chan);
        sav2csv(chan,['3Dvert_',filename],pathnames);
        sav2csv(chan1,['flat_',filename],pathnames);
        t_out = vertcat(t_out,t_tmp);
        %save to ndata
        ndata(j).rotpivots = chan;
        ndata(j).flatpivots = chan1;
    end
end
close(h1);
delete(gcp('nocreate'))

%save terminal output
%[t_out] = rmv_duplicate(t_out,[1 2 3]);     %remove duplicates...does not apply here, but remember collapse!
t_out = unique(t_out,'rows');
ndata(1).termipivots = t_out;
sav2csv(t_out,'rotated_termini',pathnames);
%localization for macs
if ispc
    slash = '\';        %Windows directory marker
else
    slash = '/';        %Mac directory marker
end
save([pathnames,slash,'ndata.mat'],'ndata');      %save ndata as well

%--------------------------------------------------------------------------
function [chan,t_out,o_out,flprot_idx] = syn_align_loco(ori,termi,third)
%Translate all the points first
o_tmp = zeros(size(ori,1),3); %translate ori to 0,0,0.  This is assuming that all vertices are positive, which for an image should be true.
t_tmp = termi(:,1:3)-ori(:,1:3);    %translate termi
%translate third
parfor j = 1:size(ori,1)
    curr_cell = third{ori(j,4)};    %find the current cell in the third channel.  Use the ordinals to pick the cell
    c_tmp{j,1} = curr_cell(:,1:3)-repmat(ori(j,1:3),size(curr_cell,1),1);       %translate channels
end

%fix one last thing, rotate all negative x's to positive, 180 flip,
%because we are enforcing directionality
flpx_idx = t_tmp(:,1)<0;   %create the flip index: 1 will flip and 0 will not
flpx_mtrx = ones(size(t_tmp,1),1);     %create flip matrix for x
flpx_mtrx(flpx_idx,:) = -1;
%modify termial point
t_tmp(:,1) = t_tmp(:,1).*flpx_mtrx;
t_tmp(:,2) = t_tmp(:,2).*flpx_mtrx;     %y is flipped as well for a 180 turn
%modify channel points (third points)
parfor m = 1:size(c_tmp,1)
    c_tmp{m,1}(:,1) = c_tmp{m,1}(:,1)*flpx_mtrx(m,1);      %rotate 180deg x
    c_tmp{m,1}(:,2) = c_tmp{m,1}(:,2)*flpx_mtrx(m,1);      %rotate 180deg y
end

%Now calculate the angle of rotation on the x axis
%calculate vector 1, ori - termi  rotated
v1 = horzcat(hypot(t_tmp(:,1),t_tmp(:,2)),zeros(size(t_tmp,1),1));   %rotating the syn axis to x 0, everything will be aligned to the y axis
v2 = t_tmp(:,[1 2]);      %the origial 
parfor m = 1:size(v1,1)    %calculate the dot product for each pair of vectors
    x_rot(m,1) = dot(v2(m,:),v1(m,:));     %calculate the dot product, which gives us the cos of the angle
end
%x_rot = acos(x_rot./(hypot(v1(:,1),v1(:,2)).*hypot(v2(:,1),v2(:,2)))).*180/pi;        %calculate the angles in degress.
x_rot = acosd(single(x_rot)./single(hypot(v1(:,1),v1(:,2)).*hypot(v2(:,1),v2(:,2))));        %calculate the angles in degress.
%find the clockwise rotations
neg_idx = ones(size(t_tmp,1),1);           %start the index
neg_idx(t_tmp(:,2)>0) = -1;        %clockwise rotation is -degree angle of rotation
%modify rotation angles
x_rot = x_rot.*neg_idx;         %clockwise and counter clockwise set.
%rotate the base vector
parfor m = 1:size(x_rot,1)
    x_tmp(m,:) = ([cosd(x_rot(m,1)),-sind(x_rot(m,1));sind(x_rot(m,1)),cosd(x_rot(m,1))]*[t_tmp(m,1);t_tmp(m,2)])';
end
t_tmp(:,1) = x_tmp(:,1);
%calculte the angle of roation on the z axis
v3 = horzcat(hypot(t_tmp(:,1),t_tmp(:,3)),zeros(size(t_tmp,1),1));   %rotating the syn axis to z 0
v4 = t_tmp(:,[1 3]);      %the origial vector
parfor n = 1:size(v1,1)        %Dot product per pair of vectors
    z_rot(n,1) = dot(v4(n,:),v3(n,:));     %calculate the dot product, which gives us the cos of the angle
end
%z_rot = acos(z_rot./(hypot(v3(:,1),v3(:,2)).*hypot(v4(:,1),v4(:,2)))).*180/pi;        %calculate the angles in degress.
z_rot = acosd(single(z_rot)./single(hypot(v3(:,1),v3(:,2)).*hypot(v4(:,1),v4(:,2))));        %calculate the angles in degress.
%find the clockwise rotations
neg_idx = ones(size(t_tmp,1),1);           %start the index
neg_idx(t_tmp(:,3)>0) = -1;        %clockwise rotation is -degree angle of rotation
%modify rotation angles
z_rot = z_rot.*neg_idx;         %clockwise and counter clockwise set.

%now rotate the third points
chan = [];      %initialize
%h = waitbar(0,'Rotating Vector: 0');    %initialize progress bar.
parfor k = 1:size(c_tmp,1)
    %waitbar(k/size(c_tmp,1),h,['Rotating Vector: ' num2str(k)]);   %update progress
    curr_cell = c_tmp{k,1};    %find the current cell in the third channel. 
    for l = 1:size(curr_cell,1)     %go through the vertices
        rotx_tmp = [cosd(x_rot(k,1)),-sind(x_rot(k,1));sind(x_rot(k,1)),cosd(x_rot(k,1))]*curr_cell(l,[1 2])';    %rotation in x plane
        rotz_tmp = [cosd(z_rot(k,1)),-sind(z_rot(k,1));sind(z_rot(k,1)),cosd(z_rot(k,1))]*[rotx_tmp(1) curr_cell(l,3)]';    %more acurate for x, but not z???
        chan_tmp(l,:) = [rotz_tmp(1) rotx_tmp(2) rotz_tmp(2)];        %store new rotated vertex.
    end
    chan_tmp(isnan(chan_tmp)) = 0;      %Nan is 0
    chan = vertcat(chan,chan_tmp);      %create the rotated third channel, we no longer need the structure of c_tmp or ori or termi, unbound.
    chan_tmp = [];
end
%close(h);   %close progress bar

%output termi and ori
if size(termi,2)==3     %no extra data, just the vertex
    t_out = [hypot(t_tmp(:,1),hypot(t_tmp(:,2),t_tmp(:,3))),zeros(size(t_tmp,1),2)];
    o_out = o_tmp;
else                    %extra data, concatenate into the matrix
    t_out = horzcat([t_tmp(:,1),zeros(size(t_tmp,1),1),t_tmp(:,3)],termi(:,end-(size(termi,2)-3):end));
    o_out = horzcat(o_tmp,ori(:,end-(size(ori,2)-3):end));
end
%output flprot_idx
flprot_idx = [flpx_idx x_rot z_rot];